/***************************************************************************
 * Copyright 2012 Kieker Project (http://kieker-monitoring.net)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package kieker.webgui.persistence.impl.utility;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;

import org.junit.Assert;
import org.junit.Test;

import com.google.common.io.Files;

import kieker.analysis.AnalysisController;

/**
 * Test class for {@link CloseableURLClassLoader}.
 * 
 * @author Nils Christian Ehmke
 */
public class CloseableURLClassLoaderTest {

	private static final String KIEKER_LIB = "kieker.jar";

	/**
	 * Default constructor. <b>Do not use this constructor. This is just a test class and not to be used outside a JUnit test!</b>
	 */
	public CloseableURLClassLoaderTest() {
		// No code necessary
	}

	/**
	 * A test making sure that the class loader loads classes correctly.
	 * 
	 * @throws URISyntaxException
	 *             If something went wrong.
	 * @throws IOException
	 *             If something went wrong.
	 */
	@Test
	public void testClassLoading() throws IOException, URISyntaxException {
		// Copy the kieker jar into a temporary directory
		final URL kiekerURL = Thread.currentThread().getContextClassLoader().getResource(KIEKER_LIB);
		final File tempDir = Files.createTempDir();
		final File newJar = new File(tempDir, KIEKER_LIB);

		Files.copy(new File(kiekerURL.toURI()), newJar);

		// Create the loader
		final CloseableURLClassLoader classLoader = new CloseableURLClassLoader(new URL[] { newJar.toURI().toURL() });

		// It should now be possible to load one of Kieker's classes
		try {
			classLoader.loadClass(AnalysisController.class.getCanonicalName());
		} catch (final ClassNotFoundException ex) {
			Assert.fail("Could not load class");
		}

		classLoader.close();

	}

	/**
	 * A test making sure that class loader can be closed.
	 * 
	 * @throws URISyntaxException
	 *             If something went wrong.
	 * @throws IOException
	 *             If something went wrong.
	 */
	@Test
	public void testClosing() throws IOException, URISyntaxException {
		// Copy the kieker jar into a temporary directory
		final URL kiekerURL = Thread.currentThread().getContextClassLoader().getResource(KIEKER_LIB);
		final File tempDir = Files.createTempDir();
		final File newJar = new File(tempDir, KIEKER_LIB);

		Files.copy(new File(kiekerURL.toURI()), newJar);

		// Create and close the class loader
		final CloseableURLClassLoader classLoader = new CloseableURLClassLoader(new URL[] { newJar.toURI().toURL() });
		classLoader.close();

		// It should be possible to delete the file now.
		Assert.assertTrue("Classloader does not close correctly", newJar.delete());
	}
}
