/***************************************************************************
 * Copyright 2014 Kieker Project (http://kieker-monitoring.net)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package kieker.webgui.web.beans.view;

import java.util.Stack;

import kieker.analysis.model.analysisMetaModel.MIAnalysisComponent;
import kieker.analysis.model.analysisMetaModel.MIInputPort;
import kieker.analysis.model.analysisMetaModel.MIOutputPort;
import kieker.analysis.model.analysisMetaModel.MIRepository;
import kieker.analysis.model.analysisMetaModel.MIRepositoryConnector;
import kieker.webgui.web.utility.IGraphListener;
import kieker.webgui.web.utility.command.AbstractEditorCommand;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

/**
 * This bean contains the necessary data behind an instance of the analysis editor for the undo and redo operations.<br>
 * </br>
 * 
 * The class is a Spring managed bean with view scope to make sure that one user (even in one session) can open multiple projects at a time without causing any
 * problems.
 * 
 * @author Nils Christian Ehmke
 */
@Component
@Scope("view")
public final class AnalysisEditorHistoryBean implements IGraphListener {

	private final Stack<AbstractEditorCommand> undoCommands = new Stack<AbstractEditorCommand>();
	private final Stack<AbstractEditorCommand> redoCommands = new Stack<AbstractEditorCommand>();

	/**
	 * Creates a new instance of this class. <b>Do not call this constructor manually. It will only be accessed by Spring.</b>
	 */
	public AnalysisEditorHistoryBean() {
		// No code necessary
	}

	/**
	 * Undos the last command.
	 */
	public void undoLastCommand() {
		if (!this.undoCommands.isEmpty()) {
			final AbstractEditorCommand command = this.undoCommands.pop();

			command.execute();

			this.redoCommands.push(command.invert());
		}
	}

	/**
	 * Redos the last command.
	 */
	public void redoLastCommand() {
		if (!this.redoCommands.isEmpty()) {
			final AbstractEditorCommand command = this.redoCommands.pop();

			command.execute();

			this.undoCommands.push(command.invert());
		}
	}

	/**
	 * Tells whether the undo stack is empty or not.
	 * 
	 * @return true iff the undo stack is empty.
	 */
	public boolean isUndoStackEmpty() {
		return this.undoCommands.isEmpty();
	}

	/**
	 * Tells whether the redo stack is empty or not.
	 * 
	 * @return true iff the redo stack is empty.
	 */
	public boolean isRedoStackEmpty() {
		return this.redoCommands.isEmpty();
	}

	@Override
	public void componentSelectedEvent(final MIAnalysisComponent selectedComponent) {
		// Not implemented yet
	}

	@Override
	public void componentDeletedEvent(final MIAnalysisComponent deletedComponent) {
		// Not implemented yet
	}

	@Override
	public void connectionAddedEvent(final MIRepositoryConnector sourcePort, final MIRepository targetRepo) {
		// Not implemented yet
	}

	@Override
	public void connectionAddedEvent(final MIOutputPort sourcePort, final MIInputPort targetPort) {
		// Not implemented yet
	}

	@Override
	public void connectionDeletedEvent(final MIRepositoryConnector sourcePort, final MIRepository targetRepo) {
		// Not implemented yet
	}

	@Override
	public void connectionDeletedEvent(final MIOutputPort sourcePort, final MIInputPort targetPort) {
		// Not implemented yet
	}

}
