/***************************************************************************
 * Copyright 2014 Kieker Project (http://kieker-monitoring.net)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package kieker.webgui.web.beans.application;

import java.util.Collections;
import java.util.Map;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

/**
 * This is a Spring managed bean responsible for holding a map with all available themes (look and feels). The content is static, which means that it is not possible
 * (or at least not intended) to add further themes during runtime. If you want to change the themes (or add new ones) take a look into the configuration file
 * {@code spring-bean-config.xml}. <br>
 * </br>
 * 
 * This class is a singleton scoped bean, as the list with the available themes is a static list. As this bean is Spring managed, it is <b>not</b> recommended to use
 * the constructor or the setter methods.
 * 
 * @author Nils Christian Ehmke
 */
@Component
@Scope("singleton")
public final class ThemeSwitcherBean {

	private Map<String, String> themes;

	/**
	 * Default constructor. <b>Do not use this constructor. This bean is Spring managed.</b>
	 */
	public ThemeSwitcherBean() {
		// No code necessary
	}

	public Map<String, String> getThemes() {
		return this.themes;
	}

	public void setThemes(final Map<String, String> themes) {
		// Make sure that the map with the themes is read-only.
		this.themes = Collections.unmodifiableMap(themes);
	}
}
