/***************************************************************************
 * Copyright 2014 Kieker Project (http://kieker-monitoring.net)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/
package kieker.webgui.service.impl.utility.model;

import java.util.ArrayList;
import java.util.List;

import kieker.webgui.service.impl.utility.KielerInterface;

import de.cau.cs.kieler.core.kgraph.KEdge;

/**
 * Objects of this class represent directed edges between {@link Port}s or {@link Node}s without {@link Port}s.
 * 
 * @author Robin Weiss, Florian Fittkau
 * 
 */
public class Edge {

	private final KEdge kielerEdge;
	private final Node source;
	private final Node target;
	private final Port sourcePort;
	private final Port targetPort;
	private final List<Point> bendPoints;

	/**
	 * Constructor for an {@link Edge}.
	 * 
	 * @param source
	 *            the source {@link Node} of the edge
	 * @param target
	 *            the target {@link Node} of the edge
	 * @param sourcePort
	 *            the {@link Port} of the source to which the edge is connected
	 *            or null if the source has no ports
	 * @param targetPort
	 *            the {@link Port} of the target to which the edge is connected
	 *            or null if the source has no ports
	 */
	public Edge(final Node source, final Node target, final Port sourcePort, final Port targetPort) {

		this.source = source;
		this.target = target;

		this.sourcePort = sourcePort;
		this.targetPort = targetPort;

		this.bendPoints = new ArrayList<Point>();
		this.kielerEdge = KielerInterface.makeKielerEdge(source, target, sourcePort, targetPort);
	}

	/**
	 * Adds a bend point to the edge.
	 * 
	 * @param x
	 *            the absolute x-coordinate of the bend point
	 * @param y
	 *            the absolute y-coordinate of the bend point
	 */
	public void addBendPoint(final float x, final float y) {
		this.bendPoints.add(new Point(x, y));
	}

	/**
	 * Checks to see if the source {@link Node} of the edge has {@link Port}s.
	 * 
	 * @return true if the source {@link Node} has {@link Port}s
	 */
	public boolean hasSourcePort() {
		return this.sourcePort != null;
	}

	/**
	 * Checks to see if the target {@link Node} of the edge has {@link Port}s.
	 * 
	 * @return true if the target {@link Node} has {@link Port}s
	 */
	public boolean hasTargetPort() {
		return this.targetPort != null;
	}

	public KEdge getKielerEdge() {
		return this.kielerEdge;
	}

	public Node getSource() {
		return this.source;
	}

	public Node getTarget() {
		return this.target;
	}

	public Port getSourcePort() {
		return this.sourcePort;
	}

	public Port getTargetPort() {
		return this.targetPort;
	}

	public List<Point> getBendPoints() {
		return this.bendPoints;
	}

}
