/***************************************************************************
 * Copyright 2014 Kieker Project (http://kieker-monitoring.net)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package kieker.webgui.service.impl.utility;

import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;

/**
 * This manager is responsible for locking critical code using keys. Internally we use {@link ReentrantLock}s, but make sure that they are removed once they are no
 * longer needed. The manager has been developed with a simple design. If necessary, the manager can be modified in order to provide a faster access to the locks.
 * 
 * @author Nils Christian Ehmke
 */
public final class LockManager {

	private final Map<String, Lock> locks = new ConcurrentHashMap<String, Lock>();

	/**
	 * Creates a new instance of this class.
	 */
	public LockManager() {
		// No code necessary
	}

	/**
	 * Locks the objects for the given keys in a definite order.
	 * 
	 * @param keys
	 *            The keys used for the locking.
	 */
	public void lock(final String[] keys) {
		// Make sure that the keys are in a definite order
		Arrays.sort(keys);

		// Lock all keys
		for (final String key : keys) {
			this.lock(key);
		}
	}

	/**
	 * Locks the object for the given key.
	 * 
	 * @param key
	 *            The key used for the locking.
	 */
	public void lock(final String key) {
		final Lock lock;

		synchronized (this) {
			// Get the existing lock object or create it if necessary
			if (this.locks.containsKey(key)) {
				lock = this.locks.get(key);
			} else {
				lock = new Lock();
				this.locks.put(key, lock);
			}

			// Prepare the locking
			lock.preLock();
		}

		lock.lock();
	}

	/**
	 * Unlocks the object for the given key.
	 * 
	 * @param key
	 *            The key used for the unlocking.
	 */
	public void unlock(final String key) {
		synchronized (this) {
			// Get the lock object and unlock it
			final Lock lock = this.locks.get(key);
			lock.unlock();

			// Remove the lock object if possible
			if (lock.isFree()) {
				this.locks.remove(key);
			}
		}
	}

	/**
	 * This is a helper class containing a lock object and a lock counter.
	 * 
	 * @author Nils Christian Ehmke
	 */
	private static final class Lock {

		private final ReentrantLock reentrantLock = new ReentrantLock();
		private final AtomicLong lockCounter = new AtomicLong();

		public Lock() {
			// No code necessary
		}

		public boolean isFree() {
			return this.lockCounter.get() == 0;
		}

		public void preLock() {
			this.lockCounter.incrementAndGet();
		}

		public void lock() {
			this.reentrantLock.lock();
		}

		public void unlock() {
			this.reentrantLock.unlock();
			this.lockCounter.decrementAndGet();
		}

	}
}
