/***************************************************************************
 * Copyright 2014 Kieker Project (http://kieker-monitoring.net)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/
package kieker.webgui.service;

import java.util.List;

import kieker.webgui.common.exception.DataAccessException;
import kieker.webgui.domain.User;

import org.springframework.security.access.prepost.PreAuthorize;

/**
 * This is an interface to the service which can be used to manage the available users within the system. The methods within this interface are only accessible by
 * administrators.
 * 
 * @author Nils Christian Ehmke
 */
public interface IUserService {

	/**
	 * Adds a user to the system.
	 * 
	 * @param user
	 *            The domain object used to extract the necessary data to create the user. It is assumed that all fields are filled.
	 * 
	 * @throws DataAccessException
	 *             If it was not possible to add the user to the system. Either because a constraint was violated or because the connection to the database has
	 *             somehow been damaged.
	 */
	@PreAuthorize("hasRole('Administrator')")
	public void addUser(User user) throws DataAccessException;

	/**
	 * Deletes a user from the system.
	 * 
	 * @param user
	 *            The domain object used to extract the necessary data to delete the user. It is only necessary that the name field is filled.
	 * 
	 * @throws DataAccessException
	 *             If it was not possible to delete the user. Either because a constraint was violated or because the connection to the database has somehow been
	 *             damaged.
	 */
	@PreAuthorize("hasRole('Administrator')")
	public void deleteUser(User user) throws DataAccessException;

	/**
	 * Edits a given user and changes the password as well.
	 * 
	 * @param user
	 *            The domain object used to extract the necessary data to edit the user. It is assumed that all fields are filled.
	 * 
	 * @throws DataAccessException
	 *             If it was not possible to edit the user. Either because a constraint was violated or because the connection to the database has somehow been
	 *             damaged.
	 */
	@PreAuthorize("hasRole('Administrator')")
	public void editUserWithPassword(User user) throws DataAccessException;

	/**
	 * Edits a given user, but does not change the password.
	 * 
	 * @param user
	 *            The domain object used to extract the necessary data to edit the user. It is assumed that all fields are filled (except for the password field).
	 * 
	 * @throws DataAccessException
	 *             If it was not possible to edit the user. Either because a constraint was violated or because the connection to the database has somehow been
	 *             damaged.
	 */
	@PreAuthorize("hasRole('Administrator')")
	public void editUserWithoutPassword(User user) throws DataAccessException;

	/**
	 * Delivers a list containing the available users within the system. The password fields of the returned domain objects will be empty. The list is always a fresh
	 * copy and can be modified at will.
	 * 
	 * @return A list with the available users.
	 * 
	 * @throws DataAccessException
	 *             If something went wrong during the reading. This happens probably if the connection to the database has somehow been damaged.
	 */
	@PreAuthorize("hasRole('Administrator')")
	public List<User> getUsers() throws DataAccessException;

}
