/***************************************************************************
 * Copyright 2014 Kieker Project (http://kieker-monitoring.net)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package kieker.webgui.domain.pluginDecorators;

import java.util.Map;

import kieker.analysis.model.analysisMetaModel.MIDisplay;
import kieker.analysis.model.analysisMetaModel.MIOutputPort;
import kieker.analysis.model.analysisMetaModel.MIPlugin;
import kieker.analysis.model.analysisMetaModel.MIRepositoryConnector;

import org.eclipse.emf.common.util.EList;

/**
 * This is the abstract base for all other decorators which decorate the plugins from the analysis meta model. It adds some properties and methods to an
 * {@link MIPlugin} instance. The already existing behavior of the {@link MIPlugin} instance is not modified.
 * 
 * @author Nils Christian Ehmke
 * 
 * @param <T>
 *            The type of the decorator. This can be specified by inheriting classes in order to provide a more specific interface to access the class.
 */
public abstract class AbstractPluginDecorator<T extends MIPlugin> extends AbstractAnalysisComponentDecorator<T> implements MIPlugin {

	private final Map<String, String> displaysDescriptions;

	/**
	 * Constructor to instantiate the fields of this class.
	 * 
	 * @param analysisComponent
	 *            The component wrapped by this decorator.
	 * @param propertiesDescriptions
	 *            The descriptions of the properties.
	 * @param description
	 *            The description of the wrapped component.
	 * @param dependency
	 *            The dependency description of the wrapped component.
	 * @param fullyInitialized
	 *            A flag to show whether the component has been fully initialized or not.
	 * @param displaysDescriptions
	 *            The descriptions of the displays.
	 */
	public AbstractPluginDecorator(final T analysisComponent, final Map<String, String> propertiesDescriptions, final String description, final String dependency,
			final boolean fullyInitialized, final Map<String, String> displaysDescriptions) {
		super(analysisComponent, propertiesDescriptions, description, dependency, fullyInitialized);

		this.displaysDescriptions = displaysDescriptions;
	}

	/**
	 * Delivers the description for the given display.
	 * 
	 * @param display
	 *            The display whose description should be delivered.
	 * 
	 * @return The description.
	 */
	public final String getDisplayDescription(final String display) {
		return this.displaysDescriptions.get(display);
	}

	@Override
	protected void refineComponentCopy(final T componentCopy) {
		super.refineComponentCopy(componentCopy);

		// Copy the output ports of the plugin instance
		for (final MIOutputPort outputPort : super.analysisComponent.getOutputPorts()) {
			final MIOutputPort outputPortCopy = FACTORY.createOutputPort();
			outputPortCopy.setName(outputPort.getName());
			outputPortCopy.setParent(componentCopy);
			componentCopy.getOutputPorts().add(outputPortCopy);
		}

		// Copy the repository "ports"
		for (final MIRepositoryConnector repositoryConnector : super.analysisComponent.getRepositories()) {
			final MIRepositoryConnector repositoryConnectorCopy = FACTORY.createRepositoryConnector();
			repositoryConnectorCopy.setName(repositoryConnector.getName());
			componentCopy.getRepositories().add(repositoryConnectorCopy);
		}

		// Copy the displays
		for (final MIDisplay display : super.analysisComponent.getDisplays()) {
			final MIDisplay displayCopy = FACTORY.createDisplay();
			displayCopy.setName(display.getName());
			displayCopy.setParent(componentCopy);
			componentCopy.getDisplays().add(displayCopy);
		}
	}

	@Override
	public final EList<MIRepositoryConnector> getRepositories() {
		return super.analysisComponent.getRepositories();
	}

	@Override
	public final EList<MIOutputPort> getOutputPorts() {
		return super.analysisComponent.getOutputPorts();
	}

	@Override
	public final EList<MIDisplay> getDisplays() {
		return super.analysisComponent.getDisplays();
	}

}
