/***************************************************************************
 * Copyright 2014 Kieker Project (http://kieker-monitoring.net)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package kieker.webgui.domain.pluginDecorators;

import java.lang.reflect.InvocationTargetException;
import java.util.Map;

import kieker.analysis.model.analysisMetaModel.MIAnalysisComponent;
import kieker.analysis.model.analysisMetaModel.MIAnalysisMetaModelFactory;
import kieker.analysis.model.analysisMetaModel.MIProperty;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;

/**
 * This is the abstract base for all other decorators which decorate the components from the analysis meta model. It adds some properties and methods to an
 * {@link MIAnalysisComponent} instance. The already existing behavior of the {@link MIAnalysisComponent} instance is not modified.
 * 
 * @author Nils Christian Ehmke
 * 
 * @param <T>
 *            The type of the decorator. This can be specified by inheriting classes in order to provide a more specific interface to access the class.
 */
public abstract class AbstractAnalysisComponentDecorator<T extends MIAnalysisComponent> implements MIAnalysisComponent {

	/** The singleton factory to create components. */
	protected static final MIAnalysisMetaModelFactory FACTORY = MIAnalysisMetaModelFactory.eINSTANCE;

	/** The component wrapped by this decorator. */
	protected final T analysisComponent;

	private final Map<String, String> propertiesDescriptions;
	private final String description;
	private final String dependency;
	private final boolean fullyInitialized;

	/**
	 * Constructor to instantiate the fields of this class.
	 * 
	 * @param analysisComponent
	 *            The component wrapped by this decorator.
	 * @param propertiesDescriptions
	 *            The descriptions of the properties.
	 * @param description
	 *            The description of the wrapped component.
	 * @param dependency
	 *            The dependency description of the wrapped component.
	 * @param fullyInitialized
	 *            A flag to show whether the component has been fully initialized or not.
	 */
	public AbstractAnalysisComponentDecorator(final T analysisComponent, final Map<String, String> propertiesDescriptions, final String description,
			final String dependency, final boolean fullyInitialized) {
		this.analysisComponent = analysisComponent;
		this.propertiesDescriptions = propertiesDescriptions;
		this.description = description;
		this.dependency = dependency;
		this.fullyInitialized = fullyInitialized;
	}

	public final String getDescription() {
		return this.description;
	}

	public final String getDependency() {
		return this.dependency;
	}

	public final boolean isFullyInitialized() {
		return this.fullyInitialized;
	}

	/**
	 * Delivers the description for the given property.
	 * 
	 * @param property
	 *            The property whose description should be delivered.
	 * 
	 * @return The description.
	 */
	public final String getPropertyDescription(final String property) {
		return this.propertiesDescriptions.get(property);
	}

	/**
	 * Delivers a new copy of the wrapped component.
	 * 
	 * @return A (deep) copy of the wrapped component.
	 */
	public final T newCopy() {
		final T componentCopy = this.createComponent();

		this.refineComponentCopy(componentCopy);

		return componentCopy;
	}

	/**
	 * Inheriting classes should implement this method to deliver the actual copy (without further properties) of the wrapped component.
	 * 
	 * @return A (non-deep) copy of the wrapped component.
	 */
	protected abstract T createComponent();

	/**
	 * Inheriting classes should overwrite this method in order to refine the copy of the wrapped component. The new method should call
	 * {@code super.refineComponentCopy(factory, componentCopy)} though, in order to make sure that the other properties will be copied as well.
	 * 
	 * @param componentCopy
	 *            The copy of the wrapped component, which can be refined by inheriting classes.
	 */
	protected void refineComponentCopy(final T componentCopy) {
		// Copy the properties
		for (final MIProperty property : this.analysisComponent.getProperties()) {
			final MIProperty propertyCopy = FACTORY.createProperty();
			propertyCopy.setName(property.getName());
			propertyCopy.setValue(property.getValue());
			componentCopy.getProperties().add(propertyCopy);
		}

		// Copy the remaining attributes
		componentCopy.setClassname(this.analysisComponent.getClassname());
		componentCopy.setName(this.analysisComponent.getName());
	}

	@Override
	public final TreeIterator<EObject> eAllContents() {
		return this.analysisComponent.eAllContents();
	}

	@Override
	public final EClass eClass() {
		return this.analysisComponent.eClass();
	}

	@Override
	public final EObject eContainer() {
		return this.analysisComponent.eContainer();
	}

	@Override
	public final EStructuralFeature eContainingFeature() {
		return this.analysisComponent.eContainingFeature();
	}

	@Override
	public final EReference eContainmentFeature() {
		return this.analysisComponent.eContainmentFeature();
	}

	@Override
	public final EList<EObject> eContents() {
		return this.analysisComponent.eContents();
	}

	@Override
	public final EList<EObject> eCrossReferences() {
		return this.analysisComponent.eCrossReferences();
	}

	@Override
	public final Object eGet(final EStructuralFeature arg0) {
		return this.analysisComponent.eGet(arg0);
	}

	@Override
	public final Object eGet(final EStructuralFeature arg0, final boolean arg1) {
		return this.analysisComponent.eGet(arg0, arg1);
	}

	@Override
	public final Object eInvoke(final EOperation arg0, final EList<?> arg1) throws InvocationTargetException {
		return this.analysisComponent.eInvoke(arg0, arg1);
	}

	@Override
	public final boolean eIsProxy() {
		return this.analysisComponent.eIsProxy();
	}

	@Override
	public final boolean eIsSet(final EStructuralFeature arg0) {
		return this.analysisComponent.eIsSet(arg0);
	}

	@Override
	public final Resource eResource() {
		return this.analysisComponent.eResource();
	}

	@Override
	public final void eSet(final EStructuralFeature arg0, final Object arg1) {
		this.analysisComponent.eSet(arg0, arg1);
	}

	@Override
	public final void eUnset(final EStructuralFeature arg0) {
		this.analysisComponent.eUnset(arg0);
	}

	@Override
	public final EList<Adapter> eAdapters() {
		return this.analysisComponent.eAdapters();
	}

	@Override
	public final boolean eDeliver() {
		return this.analysisComponent.eDeliver();
	}

	@Override
	public final void eNotify(final Notification arg0) {
		this.analysisComponent.eNotify(arg0);
	}

	@Override
	public final void eSetDeliver(final boolean arg0) {
		this.analysisComponent.eSetDeliver(arg0);
	}

	@Override
	public final String getName() {
		return this.analysisComponent.getName();
	}

	@Override
	public final void setName(final String value) {
		this.analysisComponent.setName(value);
	}

	@Override
	public final String getClassname() {
		return this.analysisComponent.getClassname();
	}

	@Override
	public final void setClassname(final String value) {
		this.analysisComponent.setClassname(value);
	}

	@Override
	public final EList<MIProperty> getProperties() {
		return this.analysisComponent.getProperties();
	}

	@Override
	public String getId() {
		return this.analysisComponent.getId();
	}

	@Override
	public void setId(final String value) {
		this.analysisComponent.setId(value);
	}

}
