/***************************************************************************
 * Copyright 2014 Kieker Project (http://kieker-monitoring.net)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package kieker.webgui.common;

import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

import kieker.common.logging.Log;
import kieker.common.logging.LogFactory;

import org.eclipse.emf.common.util.CommonUtil;

/**
 * This is a context listener, which will be activated during the initialization of the application. It initializes some environment properties, which are necessary
 * for the web application.<br>
 * </br>
 * 
 * The only two environment properties, which are currently set by this listener, are the property for the web application logger and the property for the EMF
 * cleaner thread. The first ones makes sure that later created analyses use the correct logger, whose messages can be intercepted by this application. If this
 * environment property is not set, the log entries of the analyses cannot be shown in the corresponding control panel. The second property makes sure that later
 * created analyses do not start a cleaner thread, which would result in further problems with the hot deployment and the GC.<br>
 * </br>
 * 
 * As this class is used by the servlet container, it is <b>not</b> recommended to use or access it in any way.
 * 
 * @author Nils Christian Ehmke
 */
public final class EnvironmentLoaderListener implements ServletContextListener {

	private static final Log LOG = LogFactory.getLog(EnvironmentLoaderListener.class);

	/**
	 * Creates a new instance of this class. <b>Do not call this constructor manually. It will only be accessed by the servlet container.</b>
	 */
	public EnvironmentLoaderListener() {
		// No code necessary
	}

	@Override
	public void contextDestroyed(final ServletContextEvent event) {
		// No code necessary
	}

	@Override
	public void contextInitialized(final ServletContextEvent event) {
		LOG.info("Starting Kieker.WebGUI environment initialization.");
		final long timeIn = System.currentTimeMillis();

		// We have to use some classes before setting the environment properties. This makes sure that the classes used within the WebGUI are initialized without
		// these environments. The later created classes use another class loader and are therefore initialized with the modified environment settings.
		this.initializeClasses();
		this.initializeProperties();

		final long timeOut = System.currentTimeMillis();
		final long timeDelta = timeOut - timeIn;

		LOG.info(String.format("Kieker.WebGUI environment initialized in %d ms.", timeDelta));
	}

	private void initializeClasses() {
		// The Log class has already been initialized. All that remains is the CommonUtil class. The following code is just dummy code, making sure that the class is
		// loaded.
		CommonUtil.internToUpperCase(null);
	}

	private void initializeProperties() {
		System.setProperty("kieker.common.logging.Log", "WEBGUI");
		System.setProperty("org.eclipse.emf.common.util.ReferenceClearingQueue", "false");
	}

}
